<?php

namespace SecretLinks;

class Encryption 
{
    private string $method;
    
    public function __construct()
    {
        $this->method = ENCRYPTION_METHOD;
    }
    
    public function encrypt(string $plainText, string $key): string
    {
        $key = $this->deriveKey($key);
        $ivLength = openssl_cipher_iv_length($this->method);
        $iv = openssl_random_pseudo_bytes($ivLength);
        
        $encrypted = openssl_encrypt(
            $plainText,
            $this->method,
            $key,
            OPENSSL_RAW_DATA,
            $iv
        );
        
        if ($encrypted === false) {
            throw new \Exception('Encryption failed');
        }
        
        $result = [
            'iv' => bin2hex($iv),
            'data' => bin2hex($encrypted),
            'key_hint' => substr(bin2hex($key), 0, 8)
        ];
        
        return base64_encode(json_encode($result));
    }
    
    public function decrypt(string $encryptedText, string $key): string
    {
        $key = $this->deriveKey($key);
        $data = base64_decode($encryptedText);
        
        if ($data === false) {
            throw new \Exception('Invalid encrypted data');
        }
        
        $ivLength = openssl_cipher_iv_length($this->method);
        $iv = substr($data, 0, $ivLength);
        $encrypted = substr($data, $ivLength);
        
        $decrypted = openssl_decrypt(
            $encrypted,
            $this->method,
            $key,
            OPENSSL_RAW_DATA,
            $iv
        );
        
        if ($decrypted === false) {
            throw new \Exception('Decryption failed');
        }
        
        return $decrypted;
    }
    
    public function generateKey(): string
    {
        return bin2hex(openssl_random_pseudo_bytes(16));
    }
    
    public function generateMessageId(): string
    {
        return bin2hex(openssl_random_pseudo_bytes(8));
    }
    
    private function deriveKey(string $userKey): string
    {
        return hash('sha256', $userKey, true);
    }
    
    public function createClientPayload(string $plainText, string $key): array
    {
        $encrypted = $this->encrypt($plainText, $key);
        $messageId = $this->generateMessageId();
        
        return [
            'id' => $messageId,
            'data' => $encrypted,
            'created' => time(),
            'algorithm' => $this->method
        ];
    }
}