class MessageCrypto {
    static async decrypt(encryptedData, key) {
        try {
            const data = atob(encryptedData);
            
            const iv = data.slice(0, 16);
            const encrypted = data.slice(16);
            
            const keyMaterial = await crypto.subtle.importKey(
                'raw',
                new TextEncoder().encode(key),
                { name: 'PBKDF2' },
                false,
                ['deriveKey']
            );
            
            const cryptoKey = await crypto.subtle.deriveKey(
                {
                    name: 'PBKDF2',
                    salt: new TextEncoder().encode('secret-links-salt'),
                    iterations: 1000,
                    hash: 'SHA-256'
                },
                keyMaterial,
                { name: 'AES-CBC', length: 256 },
                false,
                ['decrypt']
            );
            
            const ivBuffer = new Uint8Array(iv.split('').map(c => c.charCodeAt(0)));
            const encryptedBuffer = new Uint8Array(encrypted.split('').map(c => c.charCodeAt(0)));
            
            const decrypted = await crypto.subtle.decrypt(
                { name: 'AES-CBC', iv: ivBuffer },
                cryptoKey,
                encryptedBuffer
            );
            
            return new TextDecoder().decode(decrypted);
        } catch (error) {
            console.error('Decryption error:', error);
            return MessageCrypto.simpleDecrypt(encryptedData, key);
        }
    }
    
    static simpleDecrypt(encryptedData, key) {
        try {
            const data = atob(encryptedData);
            
            let decrypted = '';
            const keyHash = MessageCrypto.simpleHash(key);
            
            for (let i = 16; i < data.length; i++) {
                decrypted += String.fromCharCode(
                    data.charCodeAt(i) ^ keyHash.charCodeAt(i % keyHash.length)
                );
            }
            
            return decrypted;
        } catch (error) {
            throw new Error('Nie udało się odszyfrować wiadomości. Sprawdź czy link jest poprawny.');
        }
    }
    
    static simpleHash(str) {
        let hash = 0;
        for (let i = 0; i < str.length; i++) {
            const char = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + char;
            hash = hash & hash;
        }
        return Math.abs(hash).toString(36);
    }
}

async function loadMessage() {
    try {
        const encryptionKey = window.location.hash.substring(1);
        
        if (!encryptionKey) {
            throw new Error('Brak klucza szyfrowania. Sprawdź czy skopiowałeś cały link.');
        }
        
        const currentPath = window.location.pathname;
        const pathParts = currentPath.split('/');
        const basePath = pathParts.slice(0, -2).join('/') || '';
        const url = `${basePath}/api/message/${messageId}${isTestMode ? '?test=1' : ''}`;
        
        const response = await fetch(url);
        
        if (!response.ok) {
            throw new Error(`API Error: ${response.status}`);
        }
        
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || 'Wiadomość nie istnieje lub wygasła');
        }
        
        const decryptedContent = await decryptMessage(data.data.content, encryptionKey);
        
        showMessage(decryptedContent, data.data);
        
    } catch (error) {
        showError(error.message);
    }
}

async function decryptMessage(encryptedData, key) {
    if (typeof CryptoJS === 'undefined') {
        throw new Error('CryptoJS library not loaded');
    }
    
    const decodedData = atob(encryptedData);
    const encryptedObj = JSON.parse(decodedData);
    
    const iv = CryptoJS.enc.Hex.parse(encryptedObj.iv);
    const encrypted = CryptoJS.enc.Hex.parse(encryptedObj.data);
    
    const keyHash = CryptoJS.SHA256(key);
    
    const decrypted = CryptoJS.AES.decrypt(
        { ciphertext: encrypted },
        keyHash,
        { 
            iv: iv,
            mode: CryptoJS.mode.CBC,
            padding: CryptoJS.pad.Pkcs7
        }
    );
    
    const plaintext = decrypted.toString(CryptoJS.enc.Utf8);
    
    if (!plaintext) {
        throw new Error('Decryption failed - wrong key?');
    }
    
    return plaintext;
}

function hashKey(key) {
    let hash = '';
    for (let i = 0; i < 32; i++) {
        const charCode = key.charCodeAt(i % key.length);
        hash += String.fromCharCode((charCode * 31 + i * 17) % 256);
    }
    return hash;
}

function showMessage(content, messageData) {
    document.getElementById('loadingState').classList.add('hidden');
    document.getElementById('messageContainer').classList.remove('hidden');
    
    document.getElementById('decryptedMessage').textContent = content;
    
    if (messageData.test_mode) {
        document.getElementById('testModeBadge').classList.remove('hidden');
        document.getElementById('testModeInfo').classList.remove('hidden');
        document.getElementById('messageWarning').classList.add('hidden');
    } else {
        document.getElementById('messageWarning').classList.remove('hidden');
        initializeCountdown();
    }
    
    if (messageData.test_mode && messageData.views) {
        const viewInfo = document.createElement('div');
        viewInfo.className = 'text-sm text-gray-500 mt-4';
        viewInfo.textContent = `Wyświetlenia: ${messageData.views}`;
        document.getElementById('messageContent').appendChild(viewInfo);
    }
}

function showError(message) {
    document.getElementById('loadingState').classList.add('hidden');
    document.getElementById('errorContainer').classList.remove('hidden');
    document.getElementById('errorMessage').textContent = message;
}

let countdownInterval;

function initializeCountdown() {
    let burnSeconds = 50;
    
    if (window.burnDelay && typeof window.burnDelay === 'number') {
        burnSeconds = window.burnDelay;
    } else if (window.appConfig && window.appConfig.burnDelay) {
        burnSeconds = window.appConfig.burnDelay;
    }
    
    const countdownElement = document.getElementById('countdown');
    if (!countdownElement) {
        return;
    }
    
    countdownElement.textContent = burnSeconds;
    
    const progressBar = document.getElementById('progressBar');
    if (progressBar) {
        progressBar.style.width = '100%';
    }
    
    if (countdownInterval) {
        clearInterval(countdownInterval);
    }
    
    let remainingSeconds = burnSeconds;
    
    countdownInterval = setInterval(() => {
        remainingSeconds--;
        countdownElement.textContent = remainingSeconds;
        
        const progressBar = document.getElementById('progressBar');
        if (progressBar) {
            const percentage = (remainingSeconds / burnSeconds) * 100;
            progressBar.style.width = percentage + '%';
            
            if (percentage > 50) {
                progressBar.className = 'bg-red-500 h-2 rounded-full';
            } else if (percentage > 20) {
                progressBar.className = 'bg-orange-500 h-2 rounded-full';
            } else {
                progressBar.className = 'bg-red-600 h-2 rounded-full animate-pulse';
            }
        }
        
        if (remainingSeconds <= 3) {
            countdownElement.parentElement.classList.add('countdown-critical');
        }
        
        if (remainingSeconds <= 0) {
            clearInterval(countdownInterval);
            burnMessage();
        }
    }, 1000);
}

function burnMessage() {
    const messageContainer = document.getElementById('messageContainer');
    const destroyedContainer = document.getElementById('destroyedContainer');
    
    messageContainer.classList.add('burn-effect');
    
    setTimeout(() => {
        messageContainer.classList.add('hidden');
        destroyedContainer.classList.remove('hidden');
        destroyedContainer.classList.add('message-reveal');
    }, 3000);
}

function copyMessage() {
    const messageContent = document.getElementById('decryptedMessage').textContent;
    const copyButton = document.querySelector('button[onclick="copyMessage()"]');
    const copyIcon = copyButton.querySelector('svg');
    const originalText = copyButton.innerHTML;
    
    copyButton.innerHTML = `
        <svg class="w-4 h-4 mr-2 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/>
        </svg>
        Kopiowanie...
    `;
    
    const showSuccess = () => {
        copyButton.innerHTML = `
            <svg class="w-4 h-4 mr-2 text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
            </svg>
            Treść skopiowana!
        `;
        copyButton.classList.add('copy-success');
        
        setTimeout(() => {
            copyButton.innerHTML = originalText;
            copyButton.classList.remove('copy-success');
        }, 2000);
    };
    
    if (navigator.clipboard && window.isSecureContext) {
        navigator.clipboard.writeText(messageContent).then(() => {
            showSuccess();
        }).catch(() => {
            fallbackCopy();
        });
    } else {
        fallbackCopy();
    }
    
    function fallbackCopy() {
        try {
            const textarea = document.createElement('textarea');
            textarea.value = messageContent;
            document.body.appendChild(textarea);
            textarea.select();
            document.execCommand('copy');
            document.body.removeChild(textarea);
            showSuccess();
        } catch (error) {
            copyButton.innerHTML = `
                <svg class="w-4 h-4 mr-2 text-red-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                </svg>
                Błąd kopiowania
            `;
            setTimeout(() => {
                copyButton.innerHTML = originalText;
            }, 2000);
        }
    }
}

document.addEventListener('DOMContentLoaded', loadMessage);