const messageTextarea = document.getElementById('message');
const charCount = document.getElementById('charCount');

if (messageTextarea) {
    messageTextarea.addEventListener('input', function() {
        const length = this.value.length;
        const maxLength = messageTextarea.getAttribute('maxlength') || 10000;
        charCount.textContent = `${length} / ${maxLength}`;
        
        if (length > maxLength * 0.9) {
            charCount.classList.add('text-red-400');
        } else {
            charCount.classList.remove('text-red-400');
        }
    });
}

const createForm = document.getElementById('createForm');
if (createForm) {
    createForm.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const message = document.getElementById('message').value.trim();
        const testModeElement = document.getElementById('testMode');
        const testMode = testModeElement ? testModeElement.checked : false;
        const customExpiry = parseInt(document.getElementById('customExpiry').value);
        const csrfToken = document.getElementById('csrf_token').value;
        
        if (!message) {
            showError('Wiadomość nie może być pusta');
            return;
        }
        
        showLoadingState();
        
        try {
            const response = await fetch('./api/create', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    content: message,
                    test_mode: testMode,
                    custom_expiry: customExpiry,
                    csrf_token: csrfToken
                })
            });
            
            const data = await response.json();
            
            
            if (!response.ok) {
                throw new Error(data.error || 'Wystąpił błąd podczas tworzenia wiadomości');
            }
            
            if (data.success) {
                showSuccessState(data.data);
            } else {
                throw new Error(data.error || 'Nie udało się utworzyć wiadomości');
            }
        } catch (error) {
            showError(error.message);
        }
    });
}

function showLoadingState() {
    document.getElementById('createForm').classList.add('hidden');
    document.getElementById('loadingState').classList.remove('hidden');
    document.getElementById('successState').classList.add('hidden');
    document.getElementById('errorState').classList.add('hidden');
}

function showSuccessState(data) {
    document.getElementById('createForm').classList.add('hidden');
    document.getElementById('loadingState').classList.add('hidden');
    document.getElementById('successState').classList.remove('hidden');
    document.getElementById('errorState').classList.add('hidden');
    
    const linkInput = document.getElementById('generatedLink');
    linkInput.value = data.url;
    
    const expiryInfo = document.getElementById('expiryInfo');
    if (data.test_mode) {
        expiryInfo.textContent = 'Tryb testowy - wiadomość nie wygaśnie';
        expiryInfo.classList.add('text-yellow-400');
    } else {
        expiryInfo.textContent = `Wygasa: ${data.expires_at}`;
    }
}

function showError(message) {
    document.getElementById('createForm').classList.add('hidden');
    document.getElementById('loadingState').classList.add('hidden');
    document.getElementById('successState').classList.add('hidden');
    document.getElementById('errorState').classList.remove('hidden');
    document.getElementById('errorMessage').textContent = message;
}

async function copyToClipboard() {
    const linkInput = document.getElementById('generatedLink');
    const copyBtn = document.getElementById('copyBtn');
    const copyIcon = document.getElementById('copyIcon');
    const checkIcon = document.getElementById('checkIcon');
    const copyText = document.getElementById('copyText');
    
    try {
        if (navigator.clipboard && window.isSecureContext) {
            await navigator.clipboard.writeText(linkInput.value);
        } else {
            linkInput.select();
            document.execCommand('copy');
        }
        
        copyIcon.classList.add('hidden');
        checkIcon.classList.remove('hidden');
        copyText.textContent = 'Skopiowano!';
        copyBtn.classList.add('copy-success');
        
        setTimeout(() => {
            copyIcon.classList.remove('hidden');
            checkIcon.classList.add('hidden');
            copyText.textContent = 'Kopiuj';
            copyBtn.classList.remove('copy-success');
        }, 2000);
    } catch (err) {
        console.error('Failed to copy:', err);
        linkInput.select();
        copyText.textContent = 'Zaznacz i skopiuj';
        
        setTimeout(() => {
            copyText.textContent = 'Kopiuj';
        }, 2000);
    }
}

function closeSuccessModal() {
    const success = document.getElementById('successState');
    const form = document.getElementById('createForm');
    
    success.querySelector('.bg-gray-800').classList.add('modal-exit');
    
    setTimeout(() => {
        success.classList.add('hidden');
        success.querySelector('.bg-gray-800').classList.remove('modal-enter', 'modal-exit');
        
        form.classList.remove('hidden');
    }, 200);
    
    const textarea = document.getElementById('message');
    textarea.value = '';
    updateCharCount();
    
    const testModeCheckbox = document.getElementById('testMode');
    if (testModeCheckbox) {
        testModeCheckbox.checked = false;
    }
}

function createNewMessage() {
    closeSuccessModal();
}

function resetForm() {
    createNewMessage();
}

document.addEventListener('DOMContentLoaded', function() {
    const messageTextarea = document.getElementById('message');
    if (messageTextarea) {
        messageTextarea.focus();
    }
});