<?php

error_reporting(E_ALL);
ini_set('display_errors', 1);

function autoDetectUrl() {
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $path = dirname($_SERVER['REQUEST_URI']);
    $path = rtrim($path, '/');
    
    return $protocol . $host . $path;
}

$configExists = file_exists(__DIR__ . '/config.php');
$urlMismatch = false;

if ($configExists) {
    require_once __DIR__ . '/config.php';
    $currentUrl = autoDetectUrl();
    $configuredUrl = defined('APP_URL') ? APP_URL : '';
    
    $urlMismatch = (rtrim($configuredUrl, '/') !== rtrim($currentUrl, '/'));
}

if ($configExists && !$urlMismatch && !isset($_GET['reinstall'])) {
    ?>
    <!DOCTYPE html>
    <html lang="pl">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Secret Links - Już zainstalowane</title>
        <script src="https://cdn.tailwindcss.com"></script>
    </head>
    <body class="min-h-screen bg-gradient-to-br from-gray-900 via-gray-800 to-gray-900 p-8">
        <div class="max-w-md mx-auto">
            <div class="bg-gray-800 rounded-lg p-8 text-center">
                <div class="inline-flex items-center justify-center w-16 h-16 bg-yellow-600 rounded-full mb-4">
                    <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                    </svg>
                </div>
                <h2 class="text-2xl font-bold text-white mb-2">Aplikacja jest już zainstalowana!</h2>
                <p class="text-gray-400 mb-6">Ze względów bezpieczeństwa instalator został zablokowany.</p>
                <div class="bg-red-900/20 border border-red-500 rounded-lg p-4 mb-4">
                    <p class="text-red-400 font-medium">⚠️ USUŃ PLIK install.php Z SERWERA!</p>
                </div>
                <div class="flex gap-4">
                    <a href="index.php" class="flex-1 bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg transition-colors">
                        Przejdź do aplikacji
                    </a>
                    <a href="?reinstall=1" class="flex-1 bg-red-600 hover:bg-red-700 text-white py-2 px-4 rounded-lg transition-colors">
                        Przeinstaluj
                    </a>
                </div>
            </div>
        </div>
    </body>
    </html>
    <?php
    exit;
}

$step = $_GET['step'] ?? 'check';
$config = [];

if (file_exists(__DIR__ . '/config.php')) {
    require_once __DIR__ . '/config.php';
    $config = [
        'app_name' => defined('APP_NAME') ? APP_NAME : 'Secret Links',
        'app_url' => defined('APP_URL') ? APP_URL : '',
        'app_env' => defined('APP_ENV') ? APP_ENV : 'production',
        'message_expiry' => defined('MESSAGE_DEFAULT_EXPIRY') ? MESSAGE_DEFAULT_EXPIRY : 86400,
        'message_max_size' => defined('MESSAGE_MAX_SIZE') ? MESSAGE_MAX_SIZE : 10000,
        'burn_delay' => defined('MESSAGE_BURN_DELAY') ? MESSAGE_BURN_DELAY : 10,
        'rate_limit_requests' => defined('RATE_LIMIT_REQUESTS') ? RATE_LIMIT_REQUESTS : 10,
        'rate_limit_window' => defined('RATE_LIMIT_WINDOW') ? RATE_LIMIT_WINDOW : 60,
        'enable_test_mode' => defined('ENABLE_TEST_MODE') ? ENABLE_TEST_MODE : true
    ];
}

function checkRequirements() {
    $requirements = [
        'PHP Version >= 8.0' => version_compare(PHP_VERSION, '8.0.0', '>='),
        'OpenSSL Extension' => extension_loaded('openssl'),
        'JSON Extension' => extension_loaded('json'),
        'MB String Extension' => extension_loaded('mbstring'),
        'Root Directory Writable' => is_writable(__DIR__),
        'Storage Directory' => checkStorageDirectory(),
        'Public Directory Writable' => is_writable(__DIR__ . '/public') || !file_exists(__DIR__ . '/public'),
        'Config File Writable' => is_writable(__DIR__) || (file_exists(__DIR__ . '/config.php') && is_writable(__DIR__ . '/config.php'))
    ];
    
    return $requirements;
}

function checkStorageDirectory() {
    $storageDir = __DIR__ . '/storage';
    if (!file_exists($storageDir)) {
        if (@mkdir($storageDir, 0755, true)) {
            return true;
        }
        return false;
    }
    return is_writable($storageDir);
}

function generateConfig($data) {
    $template = '<?php

define(\'APP_NAME\', \'' . addslashes($data['app_name']) . '\');
define(\'APP_URL\', \'' . addslashes($data['app_url']) . '\');
define(\'APP_ENV\', \'' . addslashes($data['app_env']) . '\');

define(\'STORAGE_PATH\', __DIR__ . \'/storage/messages/\');
define(\'ACTIVE_MESSAGES_PATH\', STORAGE_PATH . \'active/\');
define(\'EXPIRED_MESSAGES_PATH\', STORAGE_PATH . \'expired/\');

define(\'MESSAGE_DEFAULT_EXPIRY\', ' . intval($data['message_expiry']) . ');
define(\'MESSAGE_MAX_SIZE\', ' . intval($data['message_max_size']) . ');
define(\'MESSAGE_BURN_DELAY\', ' . intval($data['burn_delay']) . ');
define(\'ENABLE_TEST_MODE\', ' . ($data['enable_test_mode'] ? 'true' : 'false') . ');

define(\'ENCRYPTION_METHOD\', \'AES-256-CBC\');
define(\'RATE_LIMIT_REQUESTS\', ' . intval($data['rate_limit_requests']) . ');
define(\'RATE_LIMIT_WINDOW\', ' . intval($data['rate_limit_window']) . ');

if (APP_ENV === \'development\') {
    error_reporting(E_ALL);
    ini_set(\'display_errors\', 1);
} else {
    error_reporting(0);
    ini_set(\'display_errors\', 0);
}

date_default_timezone_set(\'UTC\');

if (session_status() === PHP_SESSION_NONE) {
    ini_set(\'session.cookie_httponly\', 1);
    ini_set(\'session.use_only_cookies\', 1);
    if (APP_ENV === \'production\') {
        ini_set(\'session.cookie_secure\', 1);
    }
}

if (!file_exists(ACTIVE_MESSAGES_PATH)) {
    mkdir(ACTIVE_MESSAGES_PATH, 0755, true);
}
if (!file_exists(EXPIRED_MESSAGES_PATH)) {
    mkdir(EXPIRED_MESSAGES_PATH, 0755, true);
}
';
    
    return $template;
}

if ($_POST && $step === 'configure') {
    $config = [
        'app_name' => $_POST['app_name'] ?? 'Secret Links',
        'app_url' => rtrim($_POST['app_url'] ?? '', '/'),
        'app_env' => $_POST['app_env'] ?? 'production',
        'message_expiry' => intval($_POST['message_expiry'] ?? 86400),
        'message_max_size' => intval($_POST['message_max_size'] ?? 10000),
        'burn_delay' => intval($_POST['burn_delay'] ?? 10),
        'rate_limit_requests' => intval($_POST['rate_limit_requests'] ?? 10),
        'rate_limit_window' => intval($_POST['rate_limit_window'] ?? 60),
        'enable_test_mode' => isset($_POST['enable_test_mode']) && $_POST['enable_test_mode'] === '1'
    ];
    
    $configContent = generateConfig($config);
    
    if (file_put_contents(__DIR__ . '/config.php', $configContent)) {
        if (!file_exists(__DIR__ . '/storage/messages/active')) {
            mkdir(__DIR__ . '/storage/messages/active', 0755, true);
        }
        if (!file_exists(__DIR__ . '/storage/messages/expired')) {
            mkdir(__DIR__ . '/storage/messages/expired', 0755, true);
        }
        
        $step = 'complete';
    } else {
        $error = 'Nie udało się zapisać pliku konfiguracyjnego. Sprawdź uprawnienia.';
    }
}

?><!DOCTYPE html>
<html lang="pl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Secret Links - Instalacja</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        .requirement-ok { color: #10b981; }
        .requirement-fail { color: #ef4444; }
    </style>
</head>
<body class="min-h-screen bg-gradient-to-br from-gray-900 via-gray-800 to-gray-900 p-8">
    <div class="max-w-2xl mx-auto">
        <div class="text-center mb-8">
            <div class="inline-flex items-center justify-center w-16 h-16 bg-gradient-to-br from-blue-500 to-purple-600 rounded-xl mb-4">
                <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"/>
                </svg>
            </div>
            <h1 class="text-3xl font-bold text-white mb-2">Secret Links</h1>
            <p class="text-gray-400">Instalacja aplikacji</p>
        </div>

        <?php if ($step === 'check'): ?>
            <div class="bg-gray-800 rounded-lg p-6 mb-6">
                <h2 class="text-xl font-bold text-white mb-4">Sprawdzanie wymagań</h2>
                
                <?php 
                $requirements = checkRequirements();
                $allOk = true;
                foreach ($requirements as $name => $ok): 
                    if (!$ok) $allOk = false;
                ?>
                    <div class="flex justify-between items-center py-2">
                        <span class="text-gray-300"><?php echo $name; ?></span>
                        <span class="<?php echo $ok ? 'requirement-ok' : 'requirement-fail'; ?>">
                            <?php echo $ok ? '✓ OK' : '✗ BŁĄD'; ?>
                        </span>
                    </div>
                <?php endforeach; ?>
                
                <?php if ($allOk): ?>
                    <div class="mt-6">
                        <a href="?step=configure" class="w-full bg-blue-600 hover:bg-blue-700 text-white py-3 px-6 rounded-lg font-medium transition-colors inline-block text-center">
                            Przejdź do konfiguracji →
                        </a>
                    </div>
                <?php else: ?>
                    <div class="mt-6 p-4 bg-red-900/20 border border-red-500 rounded-lg">
                        <p class="text-red-400 font-medium">Nie wszystkie wymagania są spełnione!</p>
                        <p class="text-red-300 text-sm mt-1">Napraw błędy przed kontynuowaniem instalacji.</p>
                    </div>
                <?php endif; ?>
            </div>

        <?php elseif ($step === 'configure'): ?>
            <div class="bg-gray-800 rounded-lg p-6">
                <h2 class="text-xl font-bold text-white mb-4">Konfiguracja aplikacji</h2>
                
                <?php if (isset($error)): ?>
                    <div class="mb-4 p-4 bg-red-900/20 border border-red-500 rounded-lg">
                        <p class="text-red-400"><?php echo $error; ?></p>
                    </div>
                <?php endif; ?>
                
                <form method="POST" class="space-y-6">
                    <div>
                        <label class="block text-gray-300 mb-2">Nazwa aplikacji</label>
                        <input type="text" name="app_name" value="<?php echo htmlspecialchars($config['app_name'] ?? 'Secret Links'); ?>" 
                               class="w-full p-3 bg-gray-700 text-white rounded-lg border border-gray-600 focus:border-blue-500 focus:outline-none">
                    </div>
                    
                    <div>
                        <label class="block text-gray-300 mb-2">URL aplikacji</label>
                        <input type="url" name="app_url" value="<?php echo htmlspecialchars($config['app_url'] ?? autoDetectUrl()); ?>" 
                               class="w-full p-3 bg-gray-700 text-white rounded-lg border border-gray-600 focus:border-blue-500 focus:outline-none"
                               placeholder="https://example.com/secret-links">
                        <p class="text-sm text-gray-400 mt-1">Wykryto automatycznie: <?php echo autoDetectUrl(); ?></p>
                    </div>
                    
                    <div>
                        <label class="block text-gray-300 mb-2">Środowisko</label>
                        <select name="app_env" class="w-full p-3 bg-gray-700 text-white rounded-lg border border-gray-600 focus:border-blue-500 focus:outline-none">
                            <option value="production" <?php echo ($config['app_env'] ?? '') === 'production' ? 'selected' : ''; ?>>Production</option>
                            <option value="development" <?php echo ($config['app_env'] ?? '') === 'development' ? 'selected' : ''; ?>>Development</option>
                        </select>
                    </div>
                    
                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <label class="block text-gray-300 mb-2">Wygaśnięcie wiadomości (sekundy)</label>
                            <input type="number" name="message_expiry" value="<?php echo $config['message_expiry'] ?? 86400; ?>" 
                                   class="w-full p-3 bg-gray-700 text-white rounded-lg border border-gray-600 focus:border-blue-500 focus:outline-none">
                            <p class="text-xs text-gray-400 mt-1">86400 = 24 godziny</p>
                        </div>
                        
                        <div>
                            <label class="block text-gray-300 mb-2">Opóźnienie spalenia (sekundy)</label>
                            <input type="number" name="burn_delay" value="<?php echo $config['burn_delay'] ?? 10; ?>" 
                                   class="w-full p-3 bg-gray-700 text-white rounded-lg border border-gray-600 focus:border-blue-500 focus:outline-none">
                            <p class="text-xs text-gray-400 mt-1">Czas countdown przed usunięciem</p>
                        </div>
                    </div>
                    
                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <label class="block text-gray-300 mb-2">Max rozmiar wiadomości (znaki)</label>
                            <input type="number" name="message_max_size" value="<?php echo $config['message_max_size'] ?? 10000; ?>" 
                                   class="w-full p-3 bg-gray-700 text-white rounded-lg border border-gray-600 focus:border-blue-500 focus:outline-none">
                        </div>
                        
                        <div>
                            <label class="block text-gray-300 mb-2">Rate limit (żądania/minutę)</label>
                            <input type="number" name="rate_limit_requests" value="<?php echo $config['rate_limit_requests'] ?? 10; ?>" 
                                   class="w-full p-3 bg-gray-700 text-white rounded-lg border border-gray-600 focus:border-blue-500 focus:outline-none">
                        </div>
                    </div>
                    
                    <div>
                        <label class="flex items-center cursor-pointer">
                            <input type="checkbox" name="enable_test_mode" value="1" 
                                   <?php echo ($config['enable_test_mode'] ?? true) ? 'checked' : ''; ?>
                                   class="w-4 h-4 text-blue-600 bg-gray-700 border-gray-600 rounded focus:ring-blue-500 focus:ring-2">
                            <span class="ml-3 text-gray-300">
                                <span class="font-medium">Włącz tryb testowy</span>
                                <span class="text-sm text-gray-400 block">Pozwala na tworzenie wiadomości testowych, które nie znikają po odczycie</span>
                            </span>
                        </label>
                    </div>
                    
                    <button type="submit" class="w-full bg-blue-600 hover:bg-blue-700 text-white py-3 px-6 rounded-lg font-medium transition-colors">
                        Zapisz konfigurację
                    </button>
                </form>
            </div>

        <?php elseif ($step === 'complete'): ?>
            <div class="bg-gray-800 rounded-lg p-6 text-center">
                <div class="inline-flex items-center justify-center w-16 h-16 bg-green-600 rounded-full mb-4">
                    <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                    </svg>
                </div>
                
                <h2 class="text-2xl font-bold text-white mb-2">Instalacja zakończona!</h2>
                <p class="text-gray-400 mb-6">Secret Links zostało pomyślnie zainstalowane i skonfigurowane.</p>
                
                <div class="bg-red-900/30 border-2 border-red-500 rounded-lg p-6 mb-6 animate-pulse">
                    <div class="flex items-center mb-3">
                        <svg class="w-8 h-8 text-red-400 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                        </svg>
                        <h3 class="text-xl font-bold text-red-400">WAŻNE - BEZPIECZEŃSTWO!</h3>
                    </div>
                    <p class="text-red-300 font-medium mb-3">MUSISZ natychmiast usunąć plik install.php z serwera!</p>
                    <div class="bg-gray-900 rounded p-3 font-mono text-sm text-gray-300">
                        rm <?php echo __DIR__; ?>/install.php
                    </div>
                </div>
                
                <div class="space-y-4">
                    <a href="index.php" class="w-full bg-blue-600 hover:bg-blue-700 text-white py-3 px-6 rounded-lg font-medium transition-colors inline-block">
                        Przejdź do aplikacji →
                    </a>
                    
                    <div class="text-left bg-gray-900 rounded-lg p-4">
                        <h3 class="text-white font-medium mb-2">✅ Następne kroki:</h3>
                        <ul class="text-gray-400 text-sm space-y-2">
                            <li class="text-red-400 font-bold">🔴 Usuń plik install.php z serwera (KRYTYCZNE!)</li>
                            <li>🔒 Skonfiguruj HTTPS jeśli jeszcze nie masz</li>
                            <li>🤖 Automatyczne czyszczenie jest już włączone (nie potrzeba crona!)</li>
                            <li>📝 W produkcji zmień APP_ENV na 'production' w config.php</li>
                            <li>🛡️ Ustaw uprawnienia: <code class="bg-gray-800 px-1">chmod 644 config.php</code></li>
                        </ul>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <div class="text-center mt-8 text-gray-500 text-sm">
            <p>Secret Links v1.0 - Jednorazowe wiadomości</p>
        </div>
    </div>
</body>
</html>